<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        try {
            DB::beginTransaction();

            // Obtener o crear el rol ADMIN
            $adminRole = Role::findByName('ADMIN');
            if (!$adminRole) {
                $adminRole = Role::create([
                    'name' => trim(strtoupper('ADMIN')),
                    'guard_name' => 'web'
                ]);
            }

            // Lista de permisos
            $permisos = [
                'cobranzas.cobros_mensualidades.pagar',
                'cobranzas.cobros_mensualidades.exonerar',
                'cobranzas.cobros_mensualidades.enviar_ws',
                'cobranzas.cobros_mensualidades.index',
                'cobranzas.cobros_particulares.index',
                'cobranzas.historial_de_pagos_de_cobros.index',
                'cobranzas.reporte_de_vendedores.index'
            ];

            foreach ($permisos as $permiso) {
                $permission = Permission::firstOrCreate(
                    ['name' => str_replace(' ', '_', trim(strtolower($permiso))), 'guard_name' => 'web']
                );

                // Asignar el permiso al rol ADMIN
                if (!$adminRole->hasPermissionTo($permission, 'web')) {
                    $adminRole->givePermissionTo($permission);
                }
            }



            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack(); // Asegúrate de hacer rollback en caso de error
            throw $th; // Lanzar la excepción después de hacer rollback
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        try {
            DB::beginTransaction();

            // Obtener el rol ADMIN
            $adminRole = Role::findByName('ADMIN');

            if ($adminRole) {
                // Lista de permisos que se agregaron en el método up
                $permisos = [
                    'cobranzas.cobros_mensualidades.pagar',
                    'cobranzas.cobros_mensualidades.exonerar',
                    'cobranzas.cobros_mensualidades.enviar_ws',
                    'cobranzas.cobros_mensualidades.index',
                    'cobranzas.cobros_particulares.index',
                    'cobranzas.historial_de_pagos_de_cobros.index',
                    'cobranzas.reporte_de_vendedores.index'
                ];

                foreach ($permisos as $permiso) {
                    // Obtener el permiso
                    $permission = Permission::findByName($permiso);

                    if ($permission) {
                        // Eliminar la relación entre el rol ADMIN y el permiso
                        $adminRole->revokePermissionTo($permission);

                        // Eliminar el permiso si no está asociado a ningún otro rol
                        if (!Permission::where('id', $permission->id)->whereHas('roles')->exists()) {
                            $permission->delete();
                        }
                    }
                }
            }

            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack(); // Asegúrate de hacer rollback en caso de error
            throw $th; // Lanzar la excepción después de hacer rollback
        }
    }
};
