<?php

namespace App\Livewire\Proveedor;

use App\Classes\LivewireFormsComponent;
use App\Livewire\Forms\EmpresaForm;
use App\Models\Empresa;
use App\Models\Proveedor;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Locked;

final class ProveedorFormComponent extends LivewireFormsComponent
{
    use LivewireAlert;

    public EmpresaForm $form;

    #[Locked]
    public string $pageTitle = 'Crear Proveedor';

    public string $empresaSearch = '';

    public Collection $empresasResult;

    public ?string $uuid = '';

    public ?string $empresaUUID = '';

    public bool $isEdit = false;

    public function render()
    {
        if ($this->uuid && empty($this->getErrorBag()->getMessages())) {
            $this->fillForm();

            $this->pageTitle = "Editar {$this->empresa->razon}";
        }

        return view('livewire.proveedor.proveedor-form-component');
    }

    public function mount(?string $uuid = null)
    {
        $this->empresasResult = Empresa::doesntHave('proveedor')->take(100)->get(['rif', 'razon']);
        $this->uuid = $uuid;
        $this->empresaUUID = $uuid;
    }

    #[Computed]
    public function empresa()
    {
        return Empresa::select(['id', 'razon', 'rif', 'direccion', 'telefono', 'telefono_secundario', 'whatsapp', 'telegram', 'correo'])->where('uuid', $this->empresaUUID)->first();
    }

    public function updatingEmpresaSearch(string $value)
    {
        $this->empresasResult = Empresa::doesntHave('proveedor')->where('razon', 'like', "%$value%")->take(100)->get(['razon', 'rif']);
    }

    public function cargarEmpresa()
    {
        $empresa = Empresa::where('razon', 'like', $this->empresaSearch)->first();

        if ($empresa) {
            $this->empresaUUID = $empresa->uuid;

            $this->form->fill([
                'razon' => $empresa->razon,
                'rif' => $empresa->rif,
                'direccion' => $empresa->direccion,
                'telefono' => $empresa->telefono,
                'telefono_secundario' => $empresa->telefono_secundario,
                'whatsapp' => $empresa->whatsapp,
                'telegram' => $empresa->telegram,
                'correo' => $empresa->correo,
            ]);
            $this->reset('empresaSearch');
        }
    }

    public function save()
    {

        try {

            if ($this->uuid) {
                $empresa = Empresa::where('uuid', $this->uuid)->first();
                $this->form->save($empresa);
            } else {
                $empresa = $this->form->save();
            }

            if (Proveedor::where('empresa_id', $empresa->id)->first()) {
                return redirect()->route('proveedores.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Proveedor actualizado correctamente'
                    ]
                );
            } else {
                $empresa->proveedor()->create();

                // $this->form->reset();
                $this->reset('empresaUUID');
                return redirect()->route('proveedores.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Proveedor creado correctamente'
                    ]
                );
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al crear/actualizar el proveedor.', [
                'timer' => 5000,
            ]);
        }
    }

    protected function fillForm()
    {
        $this->form->fill([
            'razon' => $this->empresa->razon ?? '',
            'rif' => $this->empresa->rif ?? '',
            'direccion' => $this->empresa->direccion ?? '',
            'telefono' => substr($this->empresa->telefono ?? '', 0, 20),
            'telefono_secundario' => substr($this->empresa->telefono_secundario ?? '', 0, 20),
            'whatsapp' => substr($this->empresa->whatsapp ?? '', 0, 20),
            'telegram' => substr($this->empresa->telegram ?? '', 0, 20),
            'correo' => substr($this->empresa->correo ?? '', 0, 20),
        ]);
    }
}
