<?php

namespace App\Livewire\Mikrotik;

use App\Models\ProtocoloMikrotik;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\Rule;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class ProtocoloMikrotikComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    public string $nombreProtocolo = '';

    public int $puerto;

    #[Locked]
    public string $protocolo = '';

    #[Locked]
    public string $modalTitle = 'Agregar Protocolo Mikrotik';

    public function render()
    {
        $protocolos_mikrotiks = ProtocoloMikrotik::search($this->search)->paginate(15);

        return view('livewire.mikrotik.protocolo-mikrotik-component', ['protocolos' => $protocolos_mikrotiks]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function cancel()
    {
        $this->reset('nombreProtocolo', 'puerto', 'protocolo');
        $this->setTitleAddProtocolModal();
    }

    public function save()
    {
        if (! str()->isUuid($this->protocolo)) {
            $this->store();
        } else {
            $this->update();
        }
    }

    private function store()
    {

        $this->validate(
            [
                'nombreProtocolo' => 'required|unique:protocolos_mikrotiks,protocolo',
                'puerto' => 'required|integer|gt:0|unique:protocolos_mikrotiks,puerto',
            ]
        );

        try {

            $protocolo = ProtocoloMikrotik::create([
                'protocolo' => $this->nombreProtocolo,
                'puerto' => $this->puerto,
            ]);

            $this->reset('nombreProtocolo', 'puerto');

            $this->alert('success', 'Protocolo registrado satisfactoriamente.', [
                'timer' => 5000,
            ]);

        } catch (Exception $e) {
            if (isset($protocolo)) {
                $protocolo->forceDelete();
            }
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al crear el protocolo.');
        }
    }

    public function edit(string $uuid)
    {
        try {

            $protocolo = ProtocoloMikrotik::where('uuid', $uuid)->firstOrFail();
            $this->fill([
                'nombreProtocolo' => $protocolo->protocolo,
                'puerto' => $protocolo->puerto,
                'protocolo' => $protocolo->uuid,
            ]);

        } catch (ModelNotFoundException $e) {

            $this->alert('error', 'Protocolo no encontrado');

        } catch (Exception $e) {

            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al actualizar el protocolo');

        }
        $this->setTitleEditProtocolModal();
        $this->dispatch('show-modal');
    }

    public function confirmDelete(string $uuid)
    {
        $protocolo = ProtocoloMikrotik::withCount('mikrotiks')->where('uuid', $uuid)->firstOrFail();

        if ($protocolo->mikrotiks_count > 0) {
            $this->alert('error', 'No se puede eliminar un protocolo asociado a un mikrotik');
        } else {
            $this->protocolo = $uuid;
            $this->alert('warning', '¿Estas seguro?', [
                'position' => 'center',
                'text' => '¡Si elimina el protocolo no podra restaurarlo!',
                'showConfirmButton' => true,
                'confirmButtonText' => '¡Eliminar!',
                'showCancelButton' => true,
                'cancelButtonText' => 'Cancelar',
                'toast' => false,
                'onConfirmed' => 'confirmedDelete',
                'onDeny' => 'deniedDelete',
                'timer' => null,
                'reverseButtons' => true,
                'confirmButtonColor' => '#3085d6',
                'cancelButtonColor' => '#d33',
            ]);
        }
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            if ($this->protocolo) {
                $protocolo = ProtocoloMikrotik::where('uuid', $this->protocolo)->first();

                if ($protocolo) {
                    $protocolo->delete();
                    $this->reset('protocolo');
                    $this->alert(message: 'Protocolo eliminado con éxito');
                }
            }

        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar el protocolo');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->protocolo = '';
    }

    private function update()
    {
        try {
            $protocolo = ProtocoloMikrotik::where('uuid', $this->protocolo)->firstOrFail();

            $this->validate(
                [
                    'nombreProtocolo' => ['required', Rule::unique('protocolos_mikrotiks', 'protocolo')->ignore($protocolo)],
                    'puerto' => ['required', 'integer', 'gt:0', Rule::unique('protocolos_mikrotiks', 'puerto')->ignore($protocolo)],
                ]
            );

            $protocolo->fill([
                'protocolo' => $this->nombreProtocolo,
                'puerto' => $this->puerto,
            ]);

            $protocolo->save();

            $this->alert('success', 'Protocolo actualizado satisfactoriamente.', [
                'timer' => 5000,
            ]);
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Protocolo no encontrado');
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al actualizar el protocolo');
        }
        $this->setTitleAddProtocolModal();
        $this->cancel();
        $this->dispatch('hide-modal');
    }

    private function setTitleAddProtocolModal()
    {
        $this->modalTitle = 'Agregar Protocolo Mikrotik';
    }

    private function setTitleEditProtocolModal()
    {
        $this->modalTitle = "Editar protocolo ($this->nombreProtocolo)";
    }
}
