<?php

namespace App\Livewire\Mikrotik;

use App\Models\ModeloMikrotik;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Validation\Rule;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class ModeloMikrotikComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    public string $nombre = '';

    #[Locked]
    public string $modelo = '';

    #[Locked]
    public string $modalTitle = 'Agregar Modelo Mikrotik';

    public function render()
    {
        $modelos_mikrotiks = ModeloMikrotik::search($this->search)->paginate(15);
        if (session()->has('message')) $this->alert(session('type'), session('message'));

        return view('livewire.mikrotik.modelo-mikrotik-component', ['modelos' => $modelos_mikrotiks]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function cancel()
    {
        $this->reset('nombre', 'modelo');
        $this->setTitleAddModelModal();
    }

    public function save()
    {
        if (!str()->isUuid($this->modelo)) {
            $this->store();
        } else {
            $this->update();
        }
    }

    private function store()
    {

        $this->validate(
            [
                'nombre' => 'required|unique:modelos_mikrotiks',
            ]
        );

        try {

            $modelo = ModeloMikrotik::create([
                'nombre' => $this->nombre,
            ]);

            $this->reset('nombre');

            $this->alert('success', 'Modelo Mikrotik registrado satisfactoriamente.', [
                'timer' => 5000,
            ]);
        } catch (Exception $e) {
            if (isset($modelo)) {
                $modelo->forceDelete();
            }
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al crear el modelo de mikrotik.');
        }
    }

    public function edit(string $uuid)
    {
        try {
            $modelo = ModeloMikrotik::where('uuid', $uuid)->firstOrFail();
            $this->fill([
                'nombre' => $modelo->nombre,
                'modelo' => $modelo->uuid,
            ]);
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Modelo de mikrotik no encontrado');
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al actualizar al modelo de mikrotik');
        }
        $this->setTitleEditModelModal();
        $this->dispatch('show-modal');
    }

    public function confirmDelete(string $uuid)
    {
        $modelo = ModeloMikrotik::withCount('mikrotiks')->where('uuid', $uuid)->firstOrFail();

        if ($modelo->mikrotiks_count > 0) {
            $this->alert('error', 'No se puede eliminar un modelo asociado a un mikrotik');
        } else {
            $this->modelo = $uuid;
            $this->alert('warning', '¿Estas seguro?', [
                'position' => 'center',
                'text' => '¡Si elimina el modelo de mikrotik no podra restaurarlo!',
                'showConfirmButton' => true,
                'confirmButtonText' => '¡Eliminar!',
                'showCancelButton' => true,
                'cancelButtonText' => 'Cancelar',
                'toast' => false,
                'onConfirmed' => 'confirmedDelete',
                'onDeny' => 'deniedDelete',
                'timer' => null,
                'reverseButtons' => true,
                'confirmButtonColor' => '#3085d6',
                'cancelButtonColor' => '#d33',
            ]);
        }
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $modelo = ModeloMikrotik::where('uuid', $this->modelo)->first();

            if ($modelo) {
                if ($modelo->mikrotiks()->count() > 0) {
                    $this->alert('error', 'No se puede eliminar el modelo porque tiene mikrotiks asociados.');
                } else {
                    if ($modelo->remotas()->count() > 0) {
                        $this->alert('error', 'No se puede eliminar el modelo porque tiene remotas asociadas.');
                    } else {


                        $modelo->delete();
                        redirect()->route('mikrotiks.modelos.index');
                        $this->alert(message: 'Modelo mikrotik eliminado con éxito');
                        $this->search = '';
                    }
                }
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar el modelo de mikrotik');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->modelo = '';
    }

    private function update()
    {
        try {
            $modelo = ModeloMikrotik::where('uuid', $this->modelo)->firstOrFail();

            $this->validate(
                [
                    'nombre' => ['required', Rule::unique('modelos_mikrotiks', 'nombre')->ignore($modelo)],
                ]
            );

            $modelo->fill([
                'nombre' => $this->nombre,
            ]);

            $modelo->save();

            $this->alert('success', 'Modelo mikrotik actualizado satisfactoriamente.', [
                'timer' => 5000,
            ]);
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Modelo mikrotik no encontrado');
        } catch (Exception $e) {
            $this->alert('error', 'Error al actualizar el modelo');
        }
        $this->setTitleAddModelModal();
        $this->cancel();
        $this->dispatch('hide-modal');
    }

    private function setTitleAddModelModal()
    {
        $this->modalTitle = 'Agregar Modelo Mikrotik';
    }

    private function setTitleEditModelModal()
    {
        $this->modalTitle = "Editar modelo ($this->nombre)";
    }
}
