<?php

namespace App\Livewire\Forms;

use App\Models\Cliente;
use App\Models\Empresa;
use App\Models\Persona;
use App\Models\Plan;
use App\Models\Remota;
use Carbon\Carbon;
use Livewire\Form;

final class RemotaForm extends Form
{
    // Informacion tecnica
    public ?string $equipo = null;

    public ?string $tipo_router = null;

    public ?string $puerto = null;

    public ?string $ip_gestion = null;

    public ?string $ubicacion = null;

    public ?string $detalle = null;

    public ?string $zona = null;

    public ?string $encargado = null;
    // Informacion de seriales
    public ?string $serial = null;

    public ?string $kit_serial = null;

    public ?string $antena_serial = null;
    // Informacion de cobro
    public ?string $dia_corte = null;

    public ?string $dia_activacion = null;

    public ?string $correo_cuenta = null;

    public ?string $password_correo_cuenta = null;

    public ?string $fecha_cuenta = null;

    // Informacion de personas
    public ?string $cliente_uuid = null;

    public ?string $proveedor_uuid = null;

    // public ?string $socio = null;

    public ?string $vendedor_uuid = null;

    public ?string $plan_uuid = null;
    public ?string $status = "";

    public ?Remota $remota = null;

    public function rules()
    {
        return [
            // Informacion tecnica
            'equipo' => ['required'],
            'tipo_router' => ['required'],
            'puerto' => ['required'],
            'ip_gestion' => ['required'],
            'ubicacion' => ['required'],
            'detalle' => ['required'],
            'zona' => ['required'],
            'encargado' => ['required'], // 'encargado' => 'required|exists:personas,uuid
            // Informacion de seriales
            'serial' => ['required'],
            'kit_serial' => ['required'],
            'antena_serial' => ['required'],
            // Informacion de cobro
            'dia_corte' => ['required', 'date'],
            'dia_activacion' => ['required', 'date'],
            'correo_cuenta' => ['required', 'email'],
            'password_correo_cuenta' => ['required'], // 'password_correo_cuenta' => 'required|confirmed
            'fecha_cuenta' => ['required', 'date'],
            // Informacion de personas
            'cliente_uuid' => ['required', 'exists:clientes,uuid'],
            'proveedor_uuid' => ['required', 'exists:empresas,uuid'],
            'plan_uuid' => ['required', 'exists:planes,uuid'],
            // 'socio' => ['required', 'exists:personas,uuid'],
            'vendedor_uuid' => ['required', 'exists:personas,uuid'],
        ];
    }

    public function messages()
    {
        return [
            // Informacion tecnica
            'equipo.required' => "El campo es obligatorio.",
            'tipo_router.required' => "El campo es obligatorio.",
            'puerto.required' => "El campo es obligatorio.",
            'ip_gestion.required' => "El campo es obligatorio.",
            // 'ip_gestion.ip' => "El campo debe ser una dirección IP válida",
            'ubicacion.required' => "El campo es obligatorio.",
            'detalle.required' => "El campo es obligatorio.",
            'zona.required' => "El campo es obligatorio.",
            'encargado.required' => "El campo es obligatorio.",
            // Informacion de seriales
            'serial.required' => "El campo es oblrigatorio.",
            'kit_serial.required' => "El campo es obligatorio.",
            'antena_serial.required' => "El campo es obligatorio.",
            // Informacion de cobro
            'dia_corte.required' => "El campo es obligatorio.",
            'dia_corte.date' => "El campo debe ser una fecha válida",
            'dia_activacion.required' => "El campo es obligatorio.",
            'dia_activacion.date' => "El campo debes ser una fecha",
            'correo_cuenta.required' => "El campo es obligatorio.",
            'correo_cuenta.email' => "El campo debe ser un correo electrónico válido",
            'password_correo_cuenta.required' => "El campo es obligatorio.",
            'fecha_cuenta.required' => "El campo es obligatorio.",
            'fecha_cuenta.date' => "El campo debe ser una fecha",
            // Informacion de personas
            'cliente_uuid.required' => "El campo es obligatorio.",
            'cliente_uuid.exists' => "Cliente proporcionado no concuerda con los registros.",
            'proveedor_uuid.required' => "El campo es obligatorio.",
            'proveedor_uuid.exists' => "Proveedor proporcionado no concuerda con los registros.",
            'plan_uuid.required' => "El campo es obligatorio.",
            'plan_uuid.exists' => "Plan proporcionado no concuerda con los registros.",
            // 'socio.required' => "El campo es obligatorio.",
            // 'socio.exists' => "Socio proporcionado no concuerda con los registros.",
            'vendedor_uuid.required' => "El campo es obligatorio.",
            'vendedor_uuid.exists' => "Vendedor proporcionado no concuerda con los registros.",
        ];
    }

    public function save(?Remota $remota = null)
    {

        $this->remota = $remota;
        $this->validate();

        if ($remota) {
            $this->update($remota);
        } else {

            return $this->store();
        }
    }

    private function store(): Remota
    {

        $remota = Remota::create([
            // Informacion tecnica
            'equipo' => $this->equipo,
            'tipo_router' => $this->tipo_router,
            'puerto' => $this->puerto,
            'ip_gestion' => $this->ip_gestion,
            'ubicacion' => $this->ubicacion,
            'detalle' => $this->detalle,
            'zona' => $this->zona,
            'encargado' => $this->encargado,
            // Informacion de seriales
            'serial' => $this->serial,
            'kit_serial' => $this->kit_serial,
            'antena_serial' => $this->antena_serial,
            // Informacion de cobro
            'dia_corte' => Carbon::createFromFormat('d/m/Y', $this->dia_corte)->format('Y-m-d'),
            'dia_activacion' => Carbon::createFromFormat('d/m/Y', $this->dia_activacion)->format('Y-m-d'),
            'correo_cuenta' => $this->correo_cuenta,
            'password_correo_cuenta' => $this->password_correo_cuenta,
            'fecha_cuenta' => Carbon::createFromFormat('d/m/Y', $this->fecha_cuenta)->format('Y-m-d'),
            // Informacion de personas
            'cliente_id' => Cliente::where('uuid', $this->cliente_uuid)->value('id'),
            'proveedor_id' => Empresa::with('proveedor')->where('uuid', $this->proveedor_uuid)->first()->proveedor->id,
            'plan_id' => Plan::where('uuid', $this->plan_uuid)->value('id'),
            // 'socio_id' => Persona::with('socio')->where('uuid', $this->socio)->first()->socio->id,
            'vendedor_id' => Persona::with('vendedor')->where('uuid', $this->vendedor_uuid)->first()->vendedor->id,
        ]);

        return $remota;
    }

    private function update(Remota $remota)
    {

        $fecha_corte = $this->dia_corte;
        $fecha_activacion = $this->dia_activacion;
        $fecha_cuenta = $this->fecha_cuenta;

        if (str_contains($fecha_corte, '/')) {
            $fecha_corte = Carbon::createFromFormat('d/m/Y', $this->dia_corte)->format('Y-m-d');
        }

        if (str_contains($fecha_activacion, '/')) {
            $fecha_activacion = Carbon::createFromFormat('d/m/Y', $this->dia_activacion)->format('Y-m-d');
        }
        if (str_contains($fecha_cuenta, '/')) {
            $fecha_cuenta = Carbon::createFromFormat('d/m/Y', $this->fecha_cuenta)->format('Y-m-d');
        }

        $remota->fill([
            // Informacion tecnica
            'equipo' => $this->equipo,
            'tipo_router' => $this->tipo_router,
            'puerto' => $this->puerto,
            'ip_gestion' => $this->ip_gestion,
            'ubicacion' => $this->ubicacion,
            'detalle' => $this->detalle,
            'zona' => $this->zona,
            'encargado' => $this->encargado,
            // Informacion de seriales
            'serial' => $this->serial,
            'kit_serial' => $this->kit_serial,
            'antena_serial' => $this->antena_serial,
            // Informacion de cobro
            'dia_corte' => $fecha_corte,
            'dia_activacion' => $fecha_activacion,
            'correo_cuenta' => $this->correo_cuenta,
            'password_correo_cuenta' => $this->password_correo_cuenta,
            'fecha_cuenta' => $fecha_cuenta,
            // Informacion de personas
            'cliente_id' => Cliente::where('uuid', $this->cliente_uuid)->value('id'),
            'proveedor_id' => Empresa::with('proveedor')->where('uuid', $this->proveedor_uuid)->first()->proveedor->id,
            'plan_id' => Plan::where('uuid', $this->plan_uuid)->value('id'),
            // 'socio_id' => Persona::with('socio')->where('uuid', $this->socio)->first()->socio->id,
            'status' => $this->status,
            'vendedor_id' => Persona::with('vendedor')->where('uuid', $this->vendedor_uuid)->first()->vendedor->id,
        ]);

        $remota->save();
    }
}
