<?php

namespace App\Livewire\Cobranza;

use App\Models\HistorialRecargas;
use App\Models\PaymentMethod;
use App\Models\Remota;
use Exception;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class PrevisionesDePagoComponent extends Component
{
    use LivewireAlert;

    public $title = 'Previsiones de pagos';

    public $filas;
    public $payment_methods;


    protected $listeners = [
        'listenersUpdatePaymentMethod' => 'updatePaymentMethod',
        'listenersRecargar' => 'recargar',
        'listenersResetear' => 'resetear',
    ];

    public function mount()
    {
        $this->filas = Remota::getAgrupadasPorCorreo();
        // dd($this->filas[0]);
        // Obtenemos los métodos de pago de tipo 'empresa' y sus remotas activas con sus planes,
        // pero solo aquellas remotas cuyo dia_corte esté dentro del mes actual

        // Primero, obtenemos el primer y último día del mes actual
        // Obtenemos el primer y último día del mes actual en formato de fecha completa (YYYY-mm-dd)
        $startOfMonth = now()->startOfMonth()->format('Y-m-d');
        $endOfMonth = now()->endOfMonth()->format('Y-m-d');

        // Obtenemos los métodos de pago de tipo 'empresa' y sus remotas activas con sus planes,
        // además contamos las remotas activas y las que no son activas para cada método de pago
        $this->payment_methods = PaymentMethod::where('tipo_metodo_pago', 'empresa')
            ->with(['remotas' => function ($query) use ($startOfMonth, $endOfMonth) {
                // Solo incluir remotas con status 'activo' y dia_corte dentro del mes actual
                $query->where('status', 'activo')
                    //   ->whereBetween('dia_corte', [$startOfMonth, $endOfMonth])
                      ->with('plan');
            }])
            ->withCount([
                // Cuenta remotas activas en el mes actual
                'remotas as remotas_activas_count' => function ($query) use ($startOfMonth, $endOfMonth) {
                    $query->where('status', 'activo');
                        // ->whereBetween('dia_corte', [$startOfMonth, $endOfMonth]);
                },
                // Cuenta remotas NO activas en el mes actual
                'remotas as remotas_no_activas_count' => function ($query) use ($startOfMonth, $endOfMonth) {
                    $query->where('status', '!=', 'activo');
                }
            ])
            ->get()
            ->map(function ($paymentMethod) {
                // Calculamos el total del costo solo de remotas activas y con dia_corte en el mes actual
                $paymentMethod->total_costo = $paymentMethod->remotas->sum(function ($remota) {
                    return $remota->plan->costo ?? 0;
                });
                // remotas_activas_count y remotas_no_activas_count ya están disponibles por withCount
                return $paymentMethod;
            })
            ->sortByDesc('total_costo');

        // dd($this->payment_me thods[6]);
    }


    public function render()
    {
        if (session()->has('message')) $this->alert(session('type'), session('message'));

        // dd($this->filas[0]);
        // dd($this->payment_methods[0]->total_costo);
        return view('livewire.cobranza.previsiones-de-pago-component');
    }



    public function recargar($correo_cuenta = null, $fecha_recarga = null, $metodos_de_pago_id = null, $total_costo_activas = null)
    {
        // dd($correo_cuenta, $fecha_recarga);

        $ids = explode(',', $metodos_de_pago_id);
        $metodo_de_pago_id = null;
        foreach ($ids as $id) {
            if (trim($id) !== "1") { // Verifica que no sea "1" después de quitar espacios
                $metodo_de_pago_id = $id;
                break;
            }
        }
        $metodo_de_pago_id = $metodo_de_pago_id ?? 1;

        try {
            DB::beginTransaction();
            // Registrar en el historial de recargas
            HistorialRecargas::create([
                'correo_de_cuenta' => $correo_cuenta,
                'fecha_de_recarga' => $fecha_recarga,
                'monto_de_recarga' => $total_costo_activas,
                'payment_method_id' => $metodo_de_pago_id,
            ]);

            // Actualizar metodo de pago cantidad registrada y sumarle
            $payment_method = PaymentMethod::find($metodo_de_pago_id);
            $payment_method->cantidad_registrada += $total_costo_activas;
            $payment_method->save();

            DB::commit();
            $this->alert('success', 'Recarga realizada correctamente.', [
                'timer' => 5000,
            ]);
            return redirect()->route('previsiones_de_pago.index');
            // $this->mount();
            // return $this->render();
        } catch (Exception $e) {
            DB::rollBack();
            dd($e->getMessage());
            $this->alert('error', 'Error al recargar cuenta .', [
                'timer' => 5000,
            ]);
        }
    }
    public function updatePaymentMethod($remota = null, $payment_method_id = null)
    {
        $remota = Remota::find($remota);


        try {
            DB::beginTransaction();
            $remota->metodo_de_pago_id = $payment_method_id;
            $remota->save();
            DB::commit();
            $this->alert('success', 'Metodo de pago actualizado correctamente.', [
                'timer' => 5000,
            ]);
            return $this->render();
        } catch (Exception $e) {
            DB::rollBack();
            // dd($e->getMessage());

            $this->alert('error', 'Error al actualizar el metodo pago.', [
                'timer' => 5000,
            ]);
        }
    }

    public function resetear()
    {
    try {
        DB::beginTransaction();

        // Get all payment methods
        $paymentMethods = PaymentMethod::all();

        // Set 'cantidad_registrada' to 0 for each payment method
        foreach ($paymentMethods as $paymentMethod) {
            $paymentMethod->cantidad_registrada = 0;
            $paymentMethod->save();
        }

        DB::commit();

        $this->alert('success', 'Todos los métodos de pago han sido reseteados correctamente.', [
            'timer' => 5000,
        ]);
        return redirect()->route('previsiones_de_pago.index');
    } catch (\Exception $e) {
        DB::rollBack();
        $this->alert('error', 'Error al resetear los métodos de pago.', [
            'timer' => 5000,
        ]);
    }
    }
}
