<?php

namespace App\Livewire\Cliente;

use App\Classes\LivewireFormsComponent;
use App\Livewire\Forms\PersonaForm;
use App\Models\Cliente;
use App\Models\Persona;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Locked;

final class ClientePersonaFormComponent extends LivewireFormsComponent
{
    use LivewireAlert;

    public PersonaForm $form;

    #[Locked]
    public string $pageTitle = 'Crear Cliente (Persona)';

    public string $personaSearch = '';

    public Collection $personasResult;

    public ?string $uuid = '';

    public ?string $personaUUID = '';

    public bool $isEdit = false;

    public function render()
    {
        if ($this->uuid && empty($this->getErrorBag()->getMessages())) {
            $this->fillForm();
            $this->isEdit = str_contains(request()->route()->uri(), 'edit');

            $this->pageTitle = "Editar {$this->persona->nombres}";
        }

        return view('livewire.cliente.cliente-persona-form-component');
    }

    public function mount(?string $uuid = null)
    {
        $this->personasResult = Persona::doesntHave('cliente')->take(100)->get(['cedula', 'nombres']);
        $this->uuid = $uuid;
        $this->personaUUID = $uuid;
    }

    #[Computed]
    public function persona()
    {
        $cliente = Cliente::where('uuid', $this->uuid)->first();
        if ($cliente) {
            return $cliente->clientable;
        }

        return Persona::select(['id', 'nombres', 'cedula', 'direccion', 'telefono', 'telefono_secundario', 'whatsapp', 'telegram', 'correo'])->where('uuid', $this->personaUUID)->first();
    }

    public function updatingPersonaSearch(string $value)
    {
        $this->personasResult = Persona::doesntHave('cliente')->where('nombres', 'like', "%$value%")->take(100)->get(['cedula', 'nombres']);
    }

    public function cargarPersona()
    {
        $persona = Persona::where('nombres', 'like', $this->personaSearch)->first();
        if ($persona) {
            $this->personaUUID = $persona->uuid;

            $this->form->fill([
                'nombres' => $persona->nombres,
                'nacionalidad' => $persona->nacionalidad ?? 'V',
                'cedula' => $persona->cedula ?? '',
                'direccion' => $persona->direccion,
                'telefono' => $persona->telefono,
                'telefono_secundario' => $persona->telefono_secundario,
                'whatsapp' => $persona->whatsapp,
                'telegram' => $persona->telegram,
                'correo' => $persona->correo,
            ]);

            $this->reset('personaSearch');
        }
    }

    public function save()
    {

        try {

            if ($this->personaUUID) {
                $cliente = Cliente::where('uuid', $this->personaUUID)->first();
                if ($cliente) {
                    $persona = $cliente->clientable;
                } else {
                    $persona = Persona::where('uuid', $this->personaUUID)->first();
                }

                $this->form->save($persona);
            } else {
                $persona = $this->form->save();
            }

            if ($persona->cliente) {
                return redirect()->route('clientes.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Cliente actualizado correctamente'
                    ]
                );
            } else {
                $persona->cliente()->save(new Cliente());


                // $this->form->reset();
                $this->reset('personaUUID');
                return redirect()->route('clientes.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Cliente creado correctamente'
                    ]
                );
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al crear/actualizar el cliente.', [
                'timer' => 5000,
            ]);
        }
    }

    protected function fillForm()
    {
        $this->form->fill([
            'nombres' => $this->persona->nombres,
            'nacionalidad' => $this->persona->nacionalidad ?? 'V',
            'cedula' => $this->persona->cedula ?? '',
            'direccion' => $this->persona->direccion,
            'telefono' => $this->persona->telefono,
            'telefono_secundario' => $this->persona->telefono_secundario,
            'whatsapp' => $this->persona->whatsapp,
            'telegram' => $this->persona->telegram,
            'correo' => $this->persona->correo,
        ]);
    }
}
