<?php

namespace App\Livewire\Cliente;

use App\Classes\LivewireFormsComponent;
use App\Livewire\Forms\EmpresaForm;
use App\Models\Cliente;
use App\Models\Empresa;
use Exception;
use Illuminate\Database\Eloquent\Collection;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Locked;

final class ClienteEmpresaFormComponent extends LivewireFormsComponent
{
    use LivewireAlert;

    public EmpresaForm $form;

    #[Locked]
    public string $pageTitle = 'Crear Cliente (Empresa)';

    public string $empresaSearch = '';

    public Collection $empresasResult;

    public ?string $uuid = '';

    public ?string $empresaUUID = '';

    public bool $isEdit = false;

    public function render()
    {
        if ($this->uuid && empty($this->getErrorBag()->getMessages())) {
            $this->fillForm();

            $this->pageTitle = "Editar {$this->empresa->razon}";
        }

        return view('livewire.cliente.cliente-empresa-form-component');
    }

    public function mount(?string $uuid = null)
    {
        $this->empresasResult = Empresa::doesntHave('cliente')->take(100)->get(['rif', 'razon']);
        $this->uuid = $uuid;
        $this->empresaUUID = $uuid;
    }

    #[Computed]
    public function empresa()
    {
        $cliente = Cliente::where('uuid', $this->uuid)->first();
        if ($cliente) {
            return $cliente->clientable;
        }

        return Empresa::select(['id', 'razon', 'rif', 'direccion', 'telefono', 'telefono_secundario', 'whatsapp', 'telegram', 'correo'])->where('uuid', $this->empresaUUID)->first();
    }

    public function updatingEmpresaSearch(string $value)
    {
        $this->empresasResult = Empresa::doesntHave('cliente')->where('razon', 'like', "%$value%")->take(100)->get(['rif', 'razon']);
    }

    public function cargarEmpresa()
    {
        $empresa = Empresa::where('razon', 'like', $this->empresaSearch)->first();

        if ($empresa) {
            $this->empresaUUID = $empresa->uuid;

            $this->form->fill([
                'razon' => $empresa->razon,
                'rif' => $empresa->rif,
                'direccion' => $empresa->direccion ?? '',
                'telefono' => $empresa->telefono ?? '',
                'telefono_secundario' => $empresa->telefono_secundario ?? '',
                'whatsapp' => $empresa->whatsapp ?? '',
                'telegram' => $empresa->telegram ?? '',
                'correo' => $empresa->correo ?? '',
            ]);
            $this->reset('empresaSearch');
        }
    }

    public function save()
    {

        try {

            if ($this->empresaUUID) {
                $cliente = Cliente::where('uuid', $this->empresaUUID)->first();
                if ($cliente) {
                    $empresa = $cliente->clientable;
                } else {
                    $empresa = Empresa::where('uuid', $this->empresaUUID)->first();
                }
                $this->form->save($empresa);
            } else {
                $empresa = $this->form->save();
            }

            if ($empresa->cliente) {
                return redirect()->route('clientes.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Cliente actualizado correctamente'
                    ]
                );
            } else {
                $empresa->cliente()->save(new Cliente());

                // $this->alert('success', 'Cliente creado satisfactoriamente.', [
                //     'timer' => 5000,
                // ]);

                // $this->form->reset();
                $this->reset('empresaUUID');
                return redirect()->route('clientes.index')->with(
                    [
                        'type' => 'success',
                        'message' => 'Cliente creado correctamente'
                    ]
                );
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al crear/actualizar el cliente.', [
                'timer' => 5000,
            ]);
        }
    }

    protected function fillForm()
    {
        $this->form->fill([
            'razon' => $this->empresa->razon,
            'rif' => $this->empresa->rif,
            'direccion' => $this->empresa->direccion,
            'telefono' => $this->empresa->telefono,
            'telefono_secundario' => $this->empresa->telefono_secundario,
            'whatsapp' => $this->empresa->whatsapp,
            'telegram' => $this->empresa->telegram,
            'correo' => $this->empresa->correo,
        ]);
    }
}
