<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class Authenticate
{
    /**
     * Maneja una solicitud entrante.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // Verificar si el usuario no está autenticado
        if (!Auth::check()) {
            // Registrar intento sin revelar detalles específicos
            Log::warning('Acceso no autorizado', [
                'ip' => $request->ip(),
                'path' => $request->path()
            ]);

            // Para API mantener el 403 como estaba originalmente
            if ($request->is('api/*') || $request->expectsJson()) {
                return abort(403);
            }

            // Guardar URL intentada solo para GET
            if ($request->isMethod('GET')) {
                session()->put('url.intended', url()->current());
            }

            return redirect()->route('drok');
        }

        // Regenerar sesión periódicamente sin exponer lógica
        $this->refreshSession($request);

        return $next($request);
    }

    /**
     * Regenera la sesión periódicamente.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    private function refreshSession(Request $request)
    {
        // Regenerar sesión cada 30 minutos
        if (!session()->has('last_active') ||
            time() - session('last_active') > 1800) {

            session()->regenerate();
            session()->put('last_active', time());
        }
    }
}
