<?php

namespace App\Http\Controllers;

use App\Models\CobrosHistorial;
use App\Models\Configuracion;
use App\Models\Factura;
use App\Models\Cobranza;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;

class ReciboCobranzaController extends Controller
{
    /**
     * Generar y descargar recibo de pago en PDF
     */
    public function generarRecibo($id)
    {
        try {
            // Buscar el cobro en el historial
            $cobro = CobrosHistorial::with([
                'cobro',
                'cliente.persona',
                'cliente.empresa',
                'remota',
                'paymentMethod'
            ])->findOrFail($id);

            // Generar número de recibo si no existe
            if (empty($cobro->numero_recibo)) {
                $cobro->numero_recibo = $this->generarNumeroRecibo($cobro->id);
                $cobro->save();
            }

            // Obtener información de la empresa (configuración)
            $empresa = $this->obtenerDatosEmpresa();

            // Preparar datos para el PDF
            $data = [
                'cobro' => $cobro,
                'cliente' => $cobro->remota 
                    ? $cobro->remota->cliente 
                    : $cobro->cliente,
                'numero_recibo' => $cobro->numero_recibo,
                'fecha_pago' => $cobro->fecha_pago,
                'monto' => $cobro->monto,
                'concepto' => $cobro->concepto,
                'metodo_pago' => $cobro->paymentMethod ? $cobro->paymentMethod->getName() : 'N/R',
                'equipo' => $cobro->remota ? $cobro->remota->equipo : 'N/A',
                'empresa' => $empresa,
            ];

            // Generar PDF
            $pdf = Pdf::loadView('pdf.recibo-cobranza', $data);
            $pdf->setPaper('letter', 'portrait');

            // Descargar PDF
            $nombreArchivo = 'Recibo_' . $cobro->numero_recibo . '.pdf';
            return $pdf->download($nombreArchivo);

        } catch (\Exception $e) {
            return redirect()->back()->with([
                'type' => 'error',
                'message' => 'Error al generar el recibo: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Generar número de recibo único
     */
    private function generarNumeroRecibo($cobroId)
    {
        $year = date('Y');
        $month = date('m');
        
        // Formato: REC-YYYYMM-ID (Ejemplo: REC-202411-00001)
        return sprintf('REC-%s%s-%05d', $year, $month, $cobroId);
    }

    /**
     * Obtener datos de la empresa desde la configuración
     */
    private function obtenerDatosEmpresa()
    {
        // Intentar obtener desde la tabla configuracion
        try {
            $config = Configuracion::first();
            
            return [
                'nombre' => $config->valor ?? config('app.name', 'VEN2VPN'),
                'rif' => $config->rif ?? 'J-00000000-0',
                'direccion' => $config->direccion ?? 'Dirección de la empresa',
                'telefono' => $config->telefono ?? '0000-0000000',
                'email' => $config->email ?? 'info@empresa.com',
            ];
        } catch (\Exception $e) {
            // Valores por defecto si no existe configuración
            return [
                'nombre' => config('app.name', 'VEN2VPN'),
                'rif' => 'J-00000000-0',
                'direccion' => 'Dirección de la empresa',
                'telefono' => '0000-0000000',
                'email' => 'info@empresa.com',
            ];
        }
    }

    /**
     * Ver recibo en el navegador (sin descargar)
     */
    public function verRecibo($id)
    {
        try {
            $cobro = CobrosHistorial::with([
                'cobro',
                'cliente.persona',
                'cliente.empresa',
                'remota',
                'paymentMethod'
            ])->findOrFail($id);

            if (empty($cobro->numero_recibo)) {
                $cobro->numero_recibo = $this->generarNumeroRecibo($cobro->id);
                $cobro->save();
            }

            $empresa = $this->obtenerDatosEmpresa();

            $data = [
                'cobro' => $cobro,
                'cliente' => $cobro->remota 
                    ? $cobro->remota->cliente 
                    : $cobro->cliente,
                'numero_recibo' => $cobro->numero_recibo,
                'fecha_pago' => $cobro->fecha_pago,
                'monto' => $cobro->monto,
                'concepto' => $cobro->concepto,
                'metodo_pago' => $cobro->paymentMethod ? $cobro->paymentMethod->getName() : 'N/R',
                'equipo' => $cobro->remota ? $cobro->remota->equipo : 'N/A',
                'empresa' => $empresa,
            ];

            $pdf = Pdf::loadView('pdf.recibo-cobranza', $data);
            $pdf->setPaper('letter', 'portrait');

            return $pdf->stream('Recibo_' . $cobro->numero_recibo . '.pdf');

        } catch (\Exception $e) {
            return redirect()->back()->with([
                'type' => 'error',
                'message' => 'Error al visualizar el recibo: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Generar factura de mensualidad (con numeración persistente)
     */
    public function generarFacturaMensualidad($id)
    {
        try {
            // Aumentar límites para generación de PDF
            ini_set('memory_limit', '512M');
            set_time_limit(300);

            // Buscar el cobro
            $cobro = Cobranza::with([
                'remota.cliente.persona',
                'remota.cliente.empresa',
                'remota.plan',
            ])->findOrFail($id);

            // Verificar o crear factura
            $factura = Factura::where('cobro_id', $id)->first();

            if (!$factura) {
                $totalFacturas = Factura::count();
                $numeroFacturaNum = 3225 + $totalFacturas + 1;
                
                $factura = Factura::create([
                    'cobro_id' => $id,
                    'numero_factura' => $numeroFacturaNum,
                    'fecha_emision' => now(),
                    'cliente_nombre' => $cobro->remota ? $cobro->remota->cliente->getClientName() : $cobro->cliente->getClientName(),
                    'equipo' => $cobro->remota ? $cobro->remota->equipo : 'N/A',
                    'plan_nombre' => $cobro->remota && $cobro->remota->plan ? $cobro->remota->plan->nombre : 'Servicio',
                    'concepto' => $cobro->concepto,
                    'monto_total' => (float)$cobro->monto,
                    'fecha_cobro' => $cobro->fecha,
                ]);
            }

            // Preparar fechas
            $fechaEmision = \Carbon\Carbon::parse($factura->fecha_emision);
            $fechaCobro = \Carbon\Carbon::parse($factura->fecha_cobro);
            $fechaVencimiento = $fechaCobro->copy()->addDays(5);

            // Preparar Imagen Logo en Base64 (Más estable para DomPDF)
            $logoBase64 = null;
            $imagePath = public_path('light-bootstrap/img/logo.png');
            if (file_exists($imagePath)) {
                $imageData = base64_encode(file_get_contents($imagePath));
                $logoBase64 = 'data:image/png;base64,' . $imageData;
            }

            $empresa = $this->obtenerDatosEmpresa();

            $data = [
                'factura' => $factura,
                'numero_factura_simple' => $factura->numero_factura,
                'fecha_emision' => $fechaEmision,
                'fecha_vencimiento' => $fechaVencimiento,
                'fecha_cobro' => $fechaCobro,
                'empresa' => $empresa,
                'logo_base64' => $logoBase64
            ];

            // Cargar y configurar PDF
            $pdf = Pdf::loadView('pdf.factura_mensualidad', $data);
            $pdf->setPaper('letter', 'portrait');
            $pdf->setOptions([
                'isHtml5ParserEnabled' => true,
                'isRemoteEnabled' => true,
                'defaultFont' => 'sans-serif'
            ]);

            return $pdf->stream('Factura_' . $factura->numero_factura . '.pdf');

        } catch (\Exception $e) {
             // Log de error opcional si hubiera log visible
             return redirect()->back()->with([
                'type' => 'error',
                'message' => 'Error crítico al generar PDF: ' . $e->getMessage()
            ]);
        }
    }
}

