<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\LoginRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;

class CustomLoginController extends Controller
{
    /**
     * Maneja la solicitud de inicio de sesión.
     *
     * @param LoginRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(LoginRequest $request)
    {
        // Implementar rate limiting para prevenir ataques de fuerza bruta
        $throttleKey = Str::lower($request->input('email')) . '|' . $request->ip();

        if (RateLimiter::tooManyAttempts($throttleKey, 5)) {
            $seconds = RateLimiter::availableIn($throttleKey);

            return back()->withErrors([
                'email' => "Demasiados intentos de inicio de sesión. Por favor, inténtelo de nuevo en {$seconds} segundos.",
            ])->onlyInput('email');
        }

        // Validación contra patrones maliciosos (SQL Injection, XSS)
        if ($this->containsMaliciousPatterns($request->email) ||
            $this->containsMaliciousPatterns($request->password)) {

            // Registrar el intento sospechoso
            Log::warning('Intento de inicio de sesión con patrones maliciosos', [
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent(),
                'email' => $request->email
            ]);

            RateLimiter::hit($throttleKey, 60 * 10); // Penalizar con 10 minutos

            return back()->withErrors([
                'email' => 'Caracteres o palabras no permitidas detectadas.',
            ])->onlyInput('email');
        }

        // No es necesario sanitizar manualmente, Laravel ya escapa los valores
        // al usarlos en consultas SQL a través del Query Builder o Eloquent

        // Intentar autenticar
        if (Auth::attempt([
            'email' => $request->email,
            'password' => $request->password
        ], $request->boolean('remember'))) {

            RateLimiter::clear($throttleKey); // Limpiar intentos fallidos

            $request->session()->regenerate(); // Previene session fixation

            // Registrar inicio de sesión exitoso
            Log::info('Autenticación completada', [
                'user_id' => Auth::id(),
                'ip' => $request->ip()
            ]);

            return redirect()->intended('/dashboard');
        }

        // Incrementar contador de intentos fallidos
        RateLimiter::hit($throttleKey, 60);

        // Registrar intento fallido
        Log::warning('Autenticación fallida', [
            'email' => $request->email,
            'ip' => $request->ip()
        ]);

        return back()->withErrors([
            'email' => 'Las credenciales no son correctas.',
        ])->onlyInput('email');
    }

    /**
     * Verifica si una cadena contiene patrones maliciosos.
     *
     * @param string|null $input
     * @return bool
     */
    private function containsMaliciousPatterns($input)
    {
        if (empty($input)) {
            return false;
        }

        // Patrones SQL Injection
        $sqlPatterns = [
            '/\bselect\b/i', '/\binsert\b/i', '/\bupdate\b/i', '/\bdelete\b/i', '/\bdrop\b/i',
            '/\bunion\b/i', '/\bwhere\b/i', '/\bfrom\b/i', '/\bhaving\b/i', '/\bgroup\b/i',
            '/\bor\s+1=1\b/i', '/\badmin\'--/i', '/\bexec\b/i', '/\bxp_cmdshell\b/i',
            '/\bsysadmin\b/i', '/\bsystem\b/i'
        ];

        // Patrones XSS
        $xssPatterns = [
            '/<script/i', '/<\/script>/i', '/javascript:/i', '/onclick/i', '/onerror/i',
            '/onload/i', '/eval\(/i', '/document\.cookie/i', '/alert\(/i'
        ];

        // Caracteres sospechosos (pero permitiendo algunos necesarios para emails)
        $suspiciousChars = [
            '/\\\\/i', // backslash
            '/--/i',   // comentario SQL
            '/;/i',    // fin de comando SQL
            '/\*\//i', // fin de comentario
            '/\/\*/i'  // inicio de comentario
        ];

        $allPatterns = array_merge($sqlPatterns, $xssPatterns, $suspiciousChars);

        foreach ($allPatterns as $pattern) {
            if (preg_match($pattern, $input)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Recarga el captcha.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function reloadCaptcha()
    {
        return response()->json(['captcha' => captcha_img()]);
    }

    /**
     * Cierra la sesión del usuario.
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function logout()
    {
        Auth::logout();
        session()->invalidate();
        session()->regenerateToken();

        return redirect()->route('drok');
    }
}
